<?php

// api/controllers/NotificationController.php

require_once __DIR__ . '/../config/database.php';
// Assurez-vous d'avoir installé le SDK Firebase Admin : composer require kreait/firebase-php
require_once __DIR__ . '/../../vendor/autoload.php';

use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification;
use Kreait\Firebase\Messaging\MessageTarget;

class NotificationController {
    private $db;
    private $messaging;

    public function __construct() {
        $this->db = (new Database())->getConnection();
        
        // Initialisez le SDK Firebase Admin une seule fois
        // Assurez-vous que le chemin vers votre fichier JSON est correct
        $serviceAccountPath = '/home/gyytmfodec/k.jnatg.org/api/config/jna-a-2520d-firebase-adminsdk.json';
        $this->messaging = (new Factory)
            ->withServiceAccount($serviceAccountPath)
            ->createMessaging();
    }

    /**
     * Point d'entrée principal pour les requêtes de notification.
     */
    public function handleRequest() {
        header("Content-Type: application/json; charset=utf-8");
        header("Access-Control-Allow-Origin: *");
        header("Access-Control-Allow-Methods: POST, OPTIONS");
        header("Access-Control-Allow-Headers: Content-Type, Authorization");

        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            http_response_code(200);
            exit();
        }

        $action = $_POST['action'] ?? json_decode(file_get_contents('php://input'), true)['action'] ?? null;

        switch ($action) {
            case 'register_token':
                $this->registerToken();
                break;
            
            // Vous pouvez ajouter d'autres actions ici si nécessaire
            // case 'send_test':
            //     $this->sendTestNotification();
            //     break;

            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'message' => 'Action non reconnue.']);
                break;
        }
    }

    /**
     * Action 1 : Enregistre ou met à jour le token FCM d'un membre.
     */
    private function registerToken() {
        $data = json_decode(file_get_contents('php://input'), true);

        if (!isset($data['membre_id']) || !isset($data['fcm_token'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Données manquantes (membre_id, fcm_token).']);
            exit();
        }

        $membreId = intval($data['membre_id']);
        $fcmToken = $data['fcm_token'];

        try {
            $sql = "UPDATE membres SET fcm_token = :fcm_token WHERE id = :membre_id";
            $stmt = $this->db->prepare($sql);
            $stmt->bindParam(':membre_id', $membreId, PDO::PARAM_INT);
            $stmt->bindParam(':fcm_token', $fcmToken, PDO::PARAM_STR);
            
            if ($stmt->execute()) {
                echo json_encode(['success' => true, 'message' => 'Token enregistré avec succès.']);
            } else {
                http_response_code(500);
                echo json_encode(['success' => false, 'message' => 'Erreur lors de l\'enregistrement du token.']);
            }
        } catch (Exception $e) {
            http_response_code(500);
            error_log("Erreur registerToken: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Erreur serveur.']);
        }
    }

    /**
     * Action 2 : Envoie une notification push à un membre spécifique.
     * C'est la fonction que vous appellerez depuis d'autres contrôleurs.
     * @param int $membreId L'ID du membre.
     * @param string $title Le titre de la notification.
     * @param string $body Le corps de la notification.
     * @param array $data Données supplémentaires (optionnel).
     * @return bool
     */
    public function sendNotificationToMember(int $membreId, string $title, string $body, array $data = []): bool {
        try {
            // 1. Récupérer le token FCM du membre
            $stmt = $this->db->prepare("SELECT fcm_token FROM membres WHERE id = ?");
            $stmt->execute([$membreId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $fcmToken = $result['fcm_token'] ?? null;

            if (!$fcmToken) {
                error_log("Aucun token FCM trouvé pour le membre ID: $membreId");
                return false;
            }

            // 2. Préparer et envoyer la notification
            $notification = Notification::create($title, $body);
            $message = CloudMessage::withTarget(MessageTarget::token($fcmToken))
                ->withNotification($notification);

            if (!empty($data)) {
                $message = $message->withData($data);
            }

            $this->messaging->send($message);
            
            error_log("Notification push envoyée avec succès au membre ID: $membreId");
            return true;

        } catch (\Kreait\Firebase\Messaging\MessageTargetInvalidException $e) {
            error_log("Token FCM invalide pour le membre ID $membreId: " . $e->getMessage());
            // Optionnel: vous pourriez vouloir supprimer le token invalide de la BDD ici
            return false;
        } catch (Exception $e) {
            error_log("Erreur lors de l'envoi de la notification au membre ID $membreId: " . $e->getMessage());
            return false;
        }
    }
}

// Pour gérer la requête directe (pour register_token)
// Les autres contrôleurs appelleront directement `new NotificationController()->sendNotificationToMember(...)`
if (basename(__FILE__) == basename($_SERVER["SCRIPT_NAME"])) {
    $controller = new NotificationController();
    $controller->handleRequest();
}